/* --------------------------------------------------------------
  SelectBox.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

/**
 * Select Widget Class
 *
 * This widget will create a simple select box and populate it with the values from
 * the entity.
 */
export default class SelectBox {
	/**
	 * Class Constructor
	 * 
	 * @param {Setting} entity Contains the setting information.
	 * @param {StyleConfiguration} styleConfiguration Reference to the current configuration object.
	 */
	constructor(entity, styleConfiguration) {
		StyleEdit.Validator.isObject(entity);
		StyleEdit.Validator.isObject(styleConfiguration);
		
		/**
		 * @type {Setting}
		 */
		this.entity = entity;
		
		/**
		 * @type {StyleConfiguration}
		 */
		this.styleConfiguration = styleConfiguration;
	}
	
	/**
	 * Initialize the widget.
	 */
	initialize() {
		const $element = $('.edit-options-container #' + this.entity.getName());
		
		$element.change(() => {
			let found = false;
			
			// Update the style configuration. 
			$.each(this.styleConfiguration.getSettings(), (index, setting) => {
				$.each(setting.getEntries(), (index, entry) => {
					if (entry.getName() === this.entity.getName()) {
						entry.setValue($element.val());
						found = true;
						return false;
					}
				});
				
				if (found) {
					return false;
				}
			});
		});
		
		$element.val(this.entity.getValue());
		
		// Add setting tags to the parent ".form-group" element.
		if (this.entity.getTags() !== null) {
			const tags = this.entity.getTags().join('|');
			$element.parents('.form-group').attr('data-tags', tags);
		}
	}
	
	/**
	 * Get the widget HTML
	 *
	 * @return {String}
	 */
	getHtml() {
		const template = $('#widget-select-template').html();
		
		// Convert the original Setting instance into a custom object in order to support 
		// object literals in in the select-type settings. The display texts should be translated. 
		
		const setting = {
			name: this.entity.getName(),
			title: this.entity.getTitle(),
			options: []
		};
		
		$.each(this.entity.getOptions(), (index, option) => {
			let entry;
			
			if (typeof option === 'object') { // The option from the JSON file is an object literal.
				const optionKey = Object.keys(option)[0];
				entry = {
					value: optionKey,
					text: StyleEdit.Language.translate(option[optionKey], 'template')
				};
			} else { // The option from the JSON file is just a string.
				entry = {
					value: option,
					text: option
				};
			}
			
			setting.options.push(entry);
		});
		
		return Mustache.render(template, setting);
	}
}
